---
title: Schedule tasks · Cloudflare Agents docs
description: An Agent can schedule tasks to be run in the future by calling
  this.schedule(when, callback, data), where when can be a delay, a Date, or a
  cron string; callback the function name to call, and data is an object of data
  to pass to the function.
lastUpdated: 2025-09-24T13:21:38.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/agents/api-reference/schedule-tasks/
  md: https://developers.cloudflare.com/agents/api-reference/schedule-tasks/index.md
---

An Agent can schedule tasks to be run in the future by calling `this.schedule(when, callback, data)`, where `when` can be a delay, a `Date`, or a cron string; `callback` the function name to call, and `data` is an object of data to pass to the function.

Scheduled tasks can do anything a request or message from a user can: make requests, query databases, send emails, read+write state: scheduled tasks can invoke any regular method on your Agent.

### Scheduling tasks

You can call `this.schedule` within any method on an Agent, and schedule tens-of-thousands of tasks per individual Agent:

* JavaScript

  ```js
  import { Agent } from "agents";


  export class SchedulingAgent extends Agent {
    async onRequest(request) {
      // Handle an incoming request
      // Schedule a task 10 minutes from now
      // Calls the "checkFlights" method
      let { taskId } = await this.schedule(600, "checkFlights", {
        flight: "DL264",
        date: "2025-02-23",
      });
      return Response.json({ taskId });
    }


    async checkFlights(data) {
      // Invoked when our scheduled task runs
      // We can also call this.schedule here to schedule another task
    }
  }
  ```

* TypeScript

  ```ts
  import { Agent } from "agents"


  export class SchedulingAgent extends Agent {
    async onRequest(request) {
      // Handle an incoming request
      // Schedule a task 10 minutes from now
      // Calls the "checkFlights" method
      let { taskId } = await this.schedule(600, "checkFlights", { flight: "DL264", date: "2025-02-23" });
      return Response.json({ taskId });
    }


    async checkFlights(data) {
      // Invoked when our scheduled task runs
      // We can also call this.schedule here to schedule another task
    }
  }
  ```

Warning

Tasks that set a callback for a method that does not exist will throw an exception: ensure that the method named in the `callback` argument of `this.schedule` exists on your `Agent` class.

You can schedule tasks in multiple ways:

* JavaScript

  ```js
  // schedule a task to run in 10 seconds
  let task = await this.schedule(10, "someTask", { message: "hello" });


  // schedule a task to run at a specific date
  let task = await this.schedule(new Date("2025-01-01"), "someTask", {});


  // schedule a task to run every 10 minutes
  let { id } = await this.schedule("*/10 * * * *", "someTask", {
    message: "hello",
  });


  // schedule a task to run every 10 minutes, but only on Mondays
  let task = await this.schedule("*/10 * * * 1", "someTask", {
    message: "hello",
  });


  // cancel a scheduled task
  this.cancelSchedule(task.id);
  ```

* TypeScript

  ```ts
  // schedule a task to run in 10 seconds
  let task = await this.schedule(10, "someTask", { message: "hello" });


  // schedule a task to run at a specific date
  let task = await this.schedule(new Date("2025-01-01"), "someTask", {});


  // schedule a task to run every 10 minutes
  let { id } = await this.schedule("*/10 * * * *", "someTask", { message: "hello" });


  // schedule a task to run every 10 minutes, but only on Mondays
  let task = await this.schedule("*/10 * * * 1", "someTask", { message: "hello" });


  // cancel a scheduled task
  this.cancelSchedule(task.id);
  ```

Calling `await this.schedule` returns a `Schedule`, which includes the task's randomly generated `id`. You can use this `id` to retrieve or cancel the task in the future. It also provides a `type` property that indicates the type of schedule, for example, one of `"scheduled" | "delayed" | "cron"`.

Maximum scheduled tasks

Each task is mapped to a row in the Agent's underlying [SQLite database](https://developers.cloudflare.com/durable-objects/api/sqlite-storage-api/), which means that each task can be up to 2 MB in size. The maximum number of tasks must be `(task_size * tasks) + all_other_state < maximum_database_size` (currently 1GB per Agent).

### Managing scheduled tasks

You can get, cancel and filter across scheduled tasks within an Agent using the scheduling API:

* JavaScript

  ```js
  // Get a specific schedule by ID
  // Returns undefined if the task does not exist
  let task = await this.getSchedule(task.id);


  // Get all scheduled tasks
  // Returns an array of Schedule objects
  let tasks = this.getSchedules();


  // Cancel a task by its ID
  // Returns true if the task was cancelled, false if it did not exist
  await this.cancelSchedule(task.id);


  // Filter for specific tasks
  // e.g. all tasks starting in the next hour
  let tasks = this.getSchedules({
    timeRange: {
      start: new Date(Date.now()),
      end: new Date(Date.now() + 60 * 60 * 1000),
    },
  });
  ```

* TypeScript

  ```ts
  // Get a specific schedule by ID
  // Returns undefined if the task does not exist
  let task = await this.getSchedule(task.id)


  // Get all scheduled tasks
  // Returns an array of Schedule objects
  let tasks = this.getSchedules();


  // Cancel a task by its ID
  // Returns true if the task was cancelled, false if it did not exist
  await this.cancelSchedule(task.id);


  // Filter for specific tasks
  // e.g. all tasks starting in the next hour
  let tasks = this.getSchedules({
    timeRange: {
      start: new Date(Date.now()),
      end: new Date(Date.now() + 60 * 60 * 1000),
    }
  });
  ```
